<?php
/*
Plugin Name: Embed Any Document Plus
Plugin URI: http://awsm.in/ead-plus-documentation/
Description: Embed Any Document WordPress plugin lets you upload and embed your documents easily in your WordPress website without any additional browser plugins like Flash or Acrobat reader. The plugin lets you choose between Google Docs Viewer and Microsoft Office Online to display your documents. 
Version: 2.0
Author: Awsm Innovations
Author URI: http://awsm.in
License: GPL V3
*/

define('EAD_PLUS', true);

class Ead_plus
{
    private static $instance = null;
    private $plugin_path;
    private $plugin_url;
    private $plugin_base;
    private $plugin_file;
    private $plugin_version;
    private $settings_slug;
    private $text_domain = 'embed-any-document-plus';
    
    /**
     * Creates or returns an instance of this class.
     */
    public static function get_instance()
    {
        // If an instance hasn't been created and set to $instance create an instance and set it to $instance.
        if (null == self::$instance) {
            self::$instance = new self;
        }
        return self::$instance;
    }
    /**
     * Pro Exists Check
     */
    public static function pro_exists()
    {
        return self::$_instance;
    }
    /**
     * Initializes the plugin by setting localization, hooks, filters, and administrative functions.
     */
    private function __construct()
    {
        
        $this->plugin_path    = plugin_dir_path(__FILE__);
        $this->plugin_url     = plugin_dir_url(__FILE__);
        $this->plugin_base    = dirname(plugin_basename(__FILE__));
        $this->plugin_file    = __FILE__;
        $this->settings_slug  = 'ead-plus-settings';
        $this->plugin_version = '2.0';
        
        //Language Support
        $this->load_plugin_textdomain();
        
        //embeddoc shortcode support
        add_shortcode('embeddoc', array( $this, 'embed_shortcode' ));
        
        //default options
        register_activation_hook($this->plugin_file, array( $this, 'defaults' ));
        
        $this->adminfunctions();
    }
    /**
     * Localisation
     */
    public function load_plugin_textdomain() {
        load_textdomain( $this->text_domain, WP_LANG_DIR . "/embed-any-document-plus/embed-any-document-plus-" . apply_filters( 'plugin_locale', get_locale(), 'embed-any-document-plus' ) . ".mo" );
        load_plugin_textdomain($this->text_domain, false, $this->plugin_base . '/language');
    }
    /**
     * Register admin Settings style
     */
    function setting_styles()
    {
        wp_register_style('embed-settings', plugins_url('css/settings.css', $this->plugin_file), false, $this->plugin_version, 'all');
        wp_enqueue_style('embed-settings');
    }
    
    /**
     * Embed any Docs Button
     */
    public function embedbutton($args = array())
    {
        
        // Check user previlage
        if (!current_user_can('edit_posts'))
            return;
        
        // Prepares button target
        $target = is_string($args) ? $args : 'content';
        
        // Prepare args
        $args = wp_parse_args($args, array(
            'target' => $target,
            'text' => __('Add Document', $this->text_domain),
            'class' => 'awsm-embed button',
            'icon' => plugins_url('images/ead-small.png', __FILE__),
            'echo' => true,
            'shortcode' => false
        ));
        
        // Prepare EAD icon
        if ($args['icon'])
            $args['icon'] = '<img src="' . $args['icon'] . '" /> ';
        
        // Print button in media column
        $button = '<a href="javascript:void(0);" class="' . $args['class'] . '" title="' . $args['text'] . '" data-target="' . $args['target'] . '" >' . $args['icon'] . $args['text'] . '</a>';
        
        // Request assets
        wp_enqueue_media();
        
        // Print/return result
        if ($args['echo'])
            echo $button;
        return $button;
    }
    
    /**
     * Admin Easy access settings link
     */
    function settingslink($links)
    {
        $settings_link = '<a href="options-general.php?page=' . $this->settings_slug . '">' . __('Settings', $this->text_domain) . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Embed Form popup
     */
    function embedpopup()
    {
        add_thickbox();
        include($this->plugin_path . 'inc/popup.php');
    }
    
    /**
     * Register admin scripts
     */
    function embed_helper()
    {
        wp_register_style('magnific-popup', plugins_url('css/magnific-popup.css', $this->plugin_file), false, '0.9.9', 'all');
        wp_register_style('embed-css', plugins_url('css/embed.css', $this->plugin_file), false, $this->plugin_version, 'all');
        wp_register_script('embed', plugins_url('js/ead.js', $this->plugin_file), array(
            'jquery'
        ), $this->plugin_version, true);
        wp_localize_script('embed', 'emebeder', $this->embedjsdata());
        wp_enqueue_style('magnific-popup');
        wp_enqueue_script('magnific-popup');
        wp_enqueue_style('embed-css');
        if (get_option('ead_drivekey') && get_option('ead_driveClient'))
            wp_enqueue_script('gapi', 'https://apis.google.com/js/api.js', '', $this->plugin_version, true);
        if (get_option('ead_dropbox'))
            wp_enqueue_script('ead-dropbox', 'https://www.dropbox.com/static/api/2/dropins.js', '', $this->plugin_version, true);
        if (get_option('ead_box'))
            wp_enqueue_script('ead-box', 'https://app.box.com/js/static/select.js', '', $this->plugin_version, true);
        wp_enqueue_script('embed');
    }
    
    /**
     * Localize array
     */
    function embedjsdata()
    {
        $jsdata = array(
            'height' => get_option('ead_height', '500px'),
            'width' => get_option('ead_width', '100%'),
            'download' => get_option('ead_download', 'none'),
            'viewer' => get_option('ead_provider', 'google'),
            'text'   => get_option('ead_text', __('Download',$this->text_domain)),
            'cache' => 0,
            'insert_text'=>__('Select', $this->text_domain),
            'ajaxurl' => admin_url('admin-ajax.php'),
            'validtypes' => $this->validembedtypes(),
            'msextension' => $this->validextensions('ms'),
            'drextension' => $this->validextensions('all'),
            'nocontent' => __('Nothing to insert', $this->text_domain),
            'nopublic' => __('The document you have chosen is a not public.', $this->text_domain) . __(' Only the owner and explicitly shared collaborators will be able to view it.', $this->text_domain),
            'invalidurl'=>__('Invalid URL', $this->text_domain),
            'from_url'=>__('From URL', $this->text_domain),
            'pluginname'=>__('Embed Any Document Plus', $this->text_domain),
            'no_api' => __('No API key', $this->text_domain),
            'driveapiKey' => false,
            'driveclientId' => false,
            'boxapikey' => false,
            'DropboxApi' => false
        );
        if (get_option('ead_drivekey') && get_option('ead_driveClient')) {
            $jsdata['driveapiKey']   = get_option('ead_drivekey');
            $jsdata['driveclientId'] = get_option('ead_driveClient');
        }
        if (get_option('ead_dropbox'))
            $jsdata['DropboxApi'] = get_option('ead_dropbox');
        if (get_option('ead_box'))
            $jsdata['boxapikey'] = get_option('ead_box');
        return $jsdata;
    }
    
    /**
     * Shortcode Functionality
     */
    function embed_shortcode($atts)
    {
        $embed            = "";
        $durl             = "";
        $default_width    = $this->sanitize_dims(get_option('ead_width', '100%'));
        $default_height   = $this->sanitize_dims(get_option('ead_height', '500px'));
        $default_provider = get_option('ead_provider', 'google');
        $default_download = get_option('ead_download', 'none');
        $default_filesize = get_option('ead_filesize', 'yes');
        $default_text     = get_option('ead_text', __('Download',$this->text_domain));
        $show             = false;
        extract(shortcode_atts(array(
            'url' => '',
            'drive' => '',
            'id'    => false,
            'width' => $default_width,
            'height' => $default_height,
            'language' => 'en',
            'text'      => __($default_text,$this->text_domain),
            'viewer' => $default_provider,
            'download' => $default_download,
            'cache' => 'on',
            'boxtheme' => 'dark'
        ), $atts));
  
        if ($url || $id):
            
            $durl        = '';
            $privatefile = '';
            if ($this->allowdownload($viewer))
                if ($download == 'alluser' or $download == 'all') {
                    $show = true;
                } elseif ($download == 'logged' AND is_user_logged_in()) {
                    $show = true;
                }
            if ($show) {
                $filesize = 0;
                $url      = esc_url($url, array(
                    'http',
                    'https'
                ));

                $fileHtml = '';

                if($show && $default_filesize=='yes'){
                    $filedata    = wp_remote_head($url);
                    if (!is_wp_error($filedata) && isset($filedata['headers']['content-length'])) {
                        $filesize = $this->human_filesize($filedata['headers']['content-length']);
                        $fileHtml = ' [' . $filesize . ']';
                    }    
                }
                
                $durl = '<p class="embed_download"><a href="' . $url . '" download >' .$text. $fileHtml . ' </a></p>';
            }

            $providerList = array( 'google', 'microsoft', 'drive', 'box' );

            if (!in_array($viewer, $providerList))
                $viewer = 'google';
            
            if ($cache == 'off' AND $viewer == 'google') {
                if (ead_url_get_param($url)) {
                    $url .= "?" . time();
                } else {
                    $url .= "&" . time();
                }
            }
            
            $url = esc_url($url, array(
                'http',
                'https'
            ));
            
            switch ($viewer) {
                case 'google':
                    $embedsrc = '//docs.google.com/viewer?url=%1$s&embedded=true&hl=%2$s';
                    $iframe   = sprintf($embedsrc, urlencode($url), esc_attr($language));
                    break;
                
                case 'microsoft':
                    $embedsrc = '//view.officeapps.live.com/op/embed.aspx?src=%1$s';
                    $iframe   = sprintf($embedsrc, urlencode($url));
                    break;
                
                case 'drive':
                    if($id){
                        $embedsrc   = '//drive.google.com/file/d/%s/preview';
                        $iframe     = sprintf($embedsrc,  $id );
                    }else{
                       $iframe = $url; 
                    }
                    break;
                
                case 'box':
                    $embedsrc = $this->boxembed($url);
                    $iframe   = sprintf($embedsrc, urlencode($boxtheme));
                    break;
            }
            $min_height   =   '';
            if($this->in_percentage($height)){
                $min_height   =   'min-height:500px;';
            }
            if($this->check_responsive($height) AND $this->check_responsive($width)){
                $iframe_style = 'style="width:100%; height:100%; border: none; position: absolute;left:0;top:0;"';
                $doc_style      = 'style="position:relative;padding-top:90%;"';
            }else{
                $iframe_style       =  sprintf('style="width:%s; height:%s; border: none;'.$min_height.'"',esc_html($width),esc_html($height));
                $doc_style      = 'style="position:relative;"';
            }

            $iframe = '<iframe src="' . $iframe . '" ' . $iframe_style . '></iframe>';
            $show      = false;
            $embed = '<div class="ead-preview"><div class="ead-document" '.  $doc_style.'>' . $iframe . $privatefile . '</div>'.$durl.'</div>';
        else:
            $embed = __('No Url Found', 'ead');
        endif;
        return $embed;
    }
    /**
     * Check value in percentage
     *
     * @since   1.2
     * @return  Int Dimenesion
     */
    function in_percentage($dim){
        if (strstr($dim, '%')) {
            return true;
        }
        return false;
    }
    /**
     * Enable Resposive
     *
     * @since   1.2
     * @return  Boolean
     */
    function check_responsive($dim){
        if (strstr($dim, '%')) {
            $dim = preg_replace("/[^0-9]*/", '', $dim);
            if ((int)$dim == 100) {
                return true;
            }
        }
        return false;
    }
    /**
     * Private File Style
     */
    function private_style()
    {
    echo '<style type="text/css">
        .ead-document{ position:relative;}
        .ead-private{ position:absolute; width: 100%; height: 100%; left:0; top:0; background:rgba(248,237,235,0.8); text-align: center;}
        .ead-lock{ display: inline-block; vertical-align: middle;max-width: 98%;}
        .ead-dummy{ display: inline-block; vertical-align: middle; height:100%; width: 1px;}
    </style>';
    }
    /**
     * Admin menu setup
     */
    public function admin_menu()
    {
        $eadsettings = add_options_page('Embed Any Document Plus', 'Embed Any Document Plus', 'manage_options', $this->settings_slug, array( $this, 'settings_page' ));
        add_action('admin_print_styles-' . $eadsettings, array( $this, 'setting_styles' ));
    }
    /**
     * Admin settings page
     */
    public function settings_page()
    {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.'));
        }
        
        include($this->plugin_path . 'inc/settings.php');
    }
    
    /**
     * Register Settings
     */
    function register_eadsettings()
    {
        register_setting('ead-settings-group', 'ead_width', array($this,'sanitize_dims'));
        register_setting('ead-settings-group', 'ead_height', array($this,'sanitize_dims'));
        register_setting('ead-settings-group', 'ead_download');
        register_setting('ead-settings-group', 'ead_filesize');
        register_setting('ead-settings-group', 'ead_provider');
        register_setting('ead-cloud-group', 'ead_drivekey');
        register_setting('ead-cloud-group', 'ead_driveClient');
        register_setting('ead-cloud-group', 'ead_dropbox');
        register_setting('ead-cloud-group', 'ead_box');
    }
    
    /**
     * Admin Functions init
     */
    function adminfunctions()
    {
        if (is_admin()) {
            add_action('admin_footer-post-new.php', array( $this, 'embedpopup' ));
            add_action('admin_footer-post.php', array( $this, 'embedpopup' )); 
            add_action('admin_print_scripts-post-new.php', array( $this, 'embed_helper' )); 
            add_action('admin_print_scripts-post.php', array( $this, 'embed_helper' )); 
            add_action('wp_head', array( $this, 'private_style' )); 
            add_action('media_buttons', array( $this, 'embedbutton' ), 1000); 
            add_action('admin_menu', array( $this, 'admin_menu' )); 
            add_action('admin_init', array( $this, 'register_eadsettings' )); 
            add_filter("plugin_action_links_" . plugin_basename(__FILE__), array( $this, 'settingslink' ));
            add_filter('upload_mimes', array( $this, 'additional_mimes' ));
        }
    }
    
    /**
     * Adds additional mimetype for meadi uploader
     */
    function additional_mimes($mimes)
    {
        return array_merge($mimes, array(
            'svg' => 'image/svg+xml',
            'ai' => 'application/postscript'
        ));
    }
    
    /**
     * To get Overlay link
     */
    function providerlink($keys, $id, $provider)
    {
        if ($this->isprovider_api($keys)) {
            $link      = 'options-general.php?page=' . $this->settings_slug . '&tab=cloud';
            $id        = "";
            $configure = '<span class="overlay"><strong>' . __('Configure', $this->text_domain) . '</strong><i></i></span>';
            $target    = 'target="_blank"';
        } else {
            $configure = '';
            $link      = '#';
            $target    = "";
        }
        echo '<a href="' . $link . '" id="' . $id . '" ' . $target . '><span><img src="' . $this->plugin_url . 'images/icon-' . strtolower($provider) . '.png" alt="Add From ' . $provider . '" />' . __('Add from ' . $provider, $this->text_domain) . $configure . '</span></a>';
    }
    
    /**
     * To initialize default options
     */
    function defaults()
    {
        $o = array(
            'ead_width' => '100%',
            'ead_height' => '500px',
            'ead_download' => 'none',
            'ead_provider' => 'google',
            'ead_mediainsert' => '1',
            'ead_filesize' =>'yes',
        );
        foreach ($o as $k => $v) {
            if (!get_option($k))
                update_option($k, $v);
        }
        return;
    }
    
    /**
     * Dropdown Builder
     *
     * @since   1.0
     * @return  String select html
     */
    function selectbuilder($name, $options, $selected = "", $class = "",$attr="")
    {
        if (is_array($options)):
            $select_html = '<select name="' . $name . '" id="' . $name . '" class="' . $class . '" '.$attr.'>';
            foreach ($options as $key => $option) {
                $selected_html ="";
                if ($key == $selected) {
                    $selected_html = ' selected="selected"';
                }
                $select_html .= '<option value="'.$key.'" ' . $selected_html . '>' . $option . '</option>';
                
            }
            echo $select_html .= '</select>';
        endif;
    }
    /**
     * Human Readable filesize
     *
     * @since   1.0
     * @return  Human readable file size
     * @note    Replaces old gde_sanitizeOpts function
     */
    function human_filesize($bytes, $decimals = 2)
    {
        $size   = array( 'B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
        $factor = floor((strlen($bytes) - 1) / 3);
        return sprintf("%.{$decimals}f ", $bytes / pow(1024, $factor)) . @$size[$factor];
    }
    /**
     * Sanitize dimensions (width, height)
     *
     * @since   1.0
     * @return  string Sanitized dimensions, or false if value is invalid
     * @note    Replaces old gde_sanitizeOpts function
     */
    function sanitize_dims($dim)
    {
        
        // remove any spacing junk
        $dim = trim(str_replace(" ", "", $dim));
        
        if (!strstr($dim, '%')) {
            $type = "px";
            $dim  = preg_replace("/[^0-9]*/", '', $dim);
        } else {
            $type = "%";
            $dim  = preg_replace("/[^0-9]*/", '', $dim);
            if ((int) $dim > 100) {
                $dim = "100";
            }
        }
        
        if ($dim) {
            return $dim . $type;
        } else {
            return false;
        }
    }
    /**
     * get box embed url
     *
     * @since   1.0
     * @return  string embed src
     */
    function boxembed($url)
    {
        $boxdata = parse_url($url);
        if ($boxdata['host'] AND $boxdata['path']) {
            return 'https://' . $boxdata['host'] . '/embed_widget/' . $boxdata['path'] . '?theme=%1$s';
        } else {
            return '';
        }
    }
 
    /**
     * Validate Source mime type
     *
     * @since   1.0
     * @return  boolean
     */
    function validmimetypes()
    {
        $mimetypes = array(
            // Text formats
            'txt|asc|c|cc|h' => 'text/plain',
            'rtx' => 'text/richtext',
            'css' => 'text/css',
            // Misc application formats
            'js' => 'application/javascript',
            'pdf' => 'application/pdf',
            'ai' => 'application/postscript',
            'tif' => 'image/tiff',
            'tiff' => 'image/tiff',
            // MS Office formats
            'doc' => 'application/msword',
            'pot|pps|ppt' => 'application/vnd.ms-powerpoint',
            'xla|xls|xlt|xlw' => 'application/vnd.ms-excel',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'dotx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
            'dotm' => 'application/vnd.ms-word.template.macroEnabled.12',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xlsm' => 'application/vnd.ms-excel.sheet.macroEnabled.12',
            'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'ppsx' => 'application/vnd.openxmlformats-officedocument.presentationml.slideshow',
            // iWork formats
            'pages' => 'application/vnd.apple.pages',
            //Additional Mime Types
            'svg' => 'image/svg+xml'
        );
        return $mimetypes;
    }
    /**
     * Checks Url Validity
     *
     * @since   1.0
     * @return  boolean
     */
    function validtype($url)
    {
        $doctypes = $this->validmimetypes();
        if (is_array($doctypes)) {
            $allowed_ext = implode("|", array_keys($doctypes));
            if (preg_match("/\.($allowed_ext)$/i", $url)) {
                return true;
            }
        } else {
            return false;
        }
    }
    /**
     * Get allowed Mime Types
     *
     * @since   1.0
     * @return  string Mimetypes
     */
    function validembedtypes()
    {
        $doctypes = $this->validmimetypes();
        return $allowedtype = implode(',', $doctypes);
    }
    /**
     * Get allowed Extensions
     *
     * @since   1.0
     * @return  string Extenstion
     */
    function validextensions($list = 'all')
    {
        $extensions['all'] = array('.css', '.js', '.pdf', '.ai', '.tif', '.tiff', '.doc', '.txt', '.asc', '.c', '.cc', '.h', '.pot', '.pps', '.ppt', '.xla', '.xls', '.xlt', '.xlw', '.docx', '.dotx', '.dotm', '.xlsx', '.xlsm', '.pptx', '.pages', '.svg', '.ppsx');
        $extensions['ms']  = array('.doc', '.pot', '.pps', '.ppt', '.xla', '.xls', '.xlt', '.xlw', '.docx', '.dotx', '.dotm', '.xlsx', '.xlsm', '.pptx', '.ppsx');
        return $allowedtype = implode(',', $extensions[$list]);
    }
    /**
     * Get allowed Mime Types for microsoft
     *
     * @since   1.0
     * @return  array Mimetypes
     */
    function microsoft_mimes()
    {
        $micro_mime = array(
            'doc' => 'application/msword',
            'pot|pps|ppt' => 'application/vnd.ms-powerpoint',
            'xla|xls|xlt|xlw' => 'application/vnd.ms-excel',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'dotx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
            'dotm' => 'application/vnd.ms-word.template.macroEnabled.12',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xlsm' => 'application/vnd.ms-excel.sheet.macroEnabled.12',
            'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation'
        );
        return $micro_mime;
    }
    /**
     * Check Allow Download
     *
     * @since   1.0
     * @return  Boolean
     */
    function allowdownload($provider)
    {
        $blacklist = array(
            'drive',
            'box'
        );
        if (in_array($provider, $blacklist)) {
            return false;
        } else {
            return true;
        }
    }
    /**
     * Check Provider API
     *
     * @since   1.0
     * @return  boolean
     */
    function isprovider_api($keys)
    {
        $itemflag = false;
        if (is_array($keys)) {
            foreach ($keys as $key) {
                if (!get_option($key)) {
                    $itemflag = true;
                    break;
                }
            }
        } else {
            if (!get_option($keys)) {
                $itemflag = true;
            }
        }
        return $itemflag;
    }
    /**
     * Get Active Menu Class
     *
     * @since   1.0
     * @return  string Class name
     */
    function getactive_menu($tab, $needle)
    {
        if ($tab == $needle) {
            echo 'nav-tab-active';
        }
    }
    
    /**
     * Checks for url query parameter
     *
     * @since   1.1
     * @return  Boolean
     */
    function url_get_param($url)
    {
        $urldata = parse_url($url);
        if (isset($urldata['query'])) {
            return false;
        } else {
            return true;
        }
    }
    
    
}

Ead_plus::get_instance();
